/*
*   Copyright (C) 2007 Kevin Krammer <kevin.krammer@gmx.at>
*
*   Permission is hereby granted, free of charge, to any person obtaining a
*   copy of this software and associated documentation files (the "Software"),
*   to deal in the Software without restriction, including without limitation
*   the rights to use, copy, modify, merge, publish, distribute, sublicense,
*   and/or sell copies of the Software, and to permit persons to whom the
*   Software is furnished to do so, subject to the following conditions:
*
*   The above copyright notice and this permission notice shall be included
*   in all copies or substantial portions of the Software.
*
*   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
*   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
*   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
*   THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
*   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
*   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
*   OTHER DEALINGS IN THE SOFTWARE.
*/

// Qt includes
#include <qdom.h>
#include <qfile.h>
#include <qstringlist.h>
#include <qtextstream.h>

// local includes
#include "classgen.h"
#include "methodgen.h"

class Set : public QMap<QString, bool>
{
public:
    void insertString(const QString& key)
    {
        insert(key, true);
    }

    void insertStringList(const QStringList& list)
    {
        QStringList::const_iterator it    = list.begin();
        QStringList::const_iterator endIt = list.end();
        for (; it != endIt; ++it)
        {
            insert(*it, true);
        }
    }
};

static void writeFileHeader(QTextStream& stream)
{
    stream << "// File autogenerated" << endl;
    stream << endl;
}

static void writeFileFooter(QTextStream& stream)
{
    stream << "// End of File" << endl;
    stream << endl;
}

static void openIncludeGuard(const QString& className, QTextStream& stream)
{
    stream << "#if !defined(" << className.upper() << "_H_INCLUDED)" << endl;
    stream << "#define " << className.upper() << "_H_INCLUDED" << endl;
    stream << endl;
}

static void closeIncludeGuard(const QString& className, QTextStream& stream)
{
    stream << "#endif //" << className.upper() << "_H_INCLUDED" << endl;
    stream << endl;
}

static void openNamespaces(const QStringList& namespaces, QTextStream& stream)
{
    QStringList::const_iterator it    = namespaces.begin();
    QStringList::const_iterator endIt = namespaces.end();
    for (; it != endIt; ++it)
    {
        stream << "namespace " << *it << endl;
        stream << "{" << endl;
    }
    stream << endl;
}

static void closeNamespaces(const QStringList& namespaces, QTextStream& stream)
{
    QStringList::const_iterator it    = namespaces.end();
    QStringList::const_iterator endIt = namespaces.end();
    for (--it; it != endIt; --it)
    {
        stream << "}; // namespace " << *it << endl;
        stream << endl;
    }
}

static void writeIncludes(const QString& description, const QStringList& includes,
        QTextStream& stream)
{
    if (includes.isEmpty()) return;

    stream << "// " << description << " includes" << endl;

    QStringList::const_iterator it    = includes.begin();
    QStringList::const_iterator endIt = includes.end();
    for (;it != endIt; ++it)
    {
        stream << "#include " << *it << endl;
    }

    stream << endl;
}

static void extractHeaderIncludes(const Method& method,
        QMap<QString, Set>& includes)
{
    QValueList<Argument>::const_iterator it    = method.arguments.begin();
    QValueList<Argument>::const_iterator endIt = method.arguments.end();
    for (; it != endIt; ++it)
    {
        if ((*it).headerIncludes.isEmpty()) continue;

        QMap<QString, QStringList>::const_iterator mapIt =
            (*it).headerIncludes.begin();
        QMap<QString, QStringList>::const_iterator mapEndIt =
            (*it).headerIncludes.end();

        for (; mapIt != mapEndIt; ++mapIt)
        {
            includes[mapIt.key()].insertStringList(mapIt.data());
        }
    }
}

static void extractForwardDeclarations(const Method& method, Set& forwards)
{
    QValueList<Argument>::const_iterator it    = method.arguments.begin();
    QValueList<Argument>::const_iterator endIt = method.arguments.end();
    for (; it != endIt; ++it)
    {
        if ((*it).forwardDeclarations.isEmpty()) continue;

        forwards.insertStringList((*it).forwardDeclarations);
    }
}

static void writeHeaderIncludes(const Class& classData, Class::Role role,
        QTextStream& stream)
{
    QMap<QString, Set> includes;
    Set forwards;

    QValueList<Method>::const_iterator it    = classData.methods.begin();
    QValueList<Method>::const_iterator endIt = classData.methods.end();
    for (; it != endIt; ++it)
    {
        if ((*it).arguments.isEmpty()) continue;

        extractHeaderIncludes(*it, includes);
        extractForwardDeclarations(*it, forwards);
    }

    QValueList<Property>::const_iterator propertyIt = classData.properties.begin();
    QValueList<Property>::const_iterator propertyEndIt = classData.properties.end();
    for (; propertyIt != propertyEndIt; ++propertyIt)
    {
        if (!(*propertyIt).headerIncludes.isEmpty())
        {
            QMap<QString, QStringList>::const_iterator mapIt =
                (*propertyIt).headerIncludes.begin();
            QMap<QString, QStringList>::const_iterator mapEndIt =
                (*propertyIt).headerIncludes.end();

            for (; mapIt != mapEndIt; ++mapIt)
            {
                includes[mapIt.key()].insertStringList(mapIt.data());
            }
        }

        if (!(*propertyIt).forwardDeclarations.isEmpty())
        {
            forwards.insertStringList((*propertyIt).forwardDeclarations);
        }
    }

    switch (role)
    {
        case Class::Interface:
            includes["qdbus"].insertString("<dbus/qdbusobject.h>");
            forwards.insertString("class QDBusError");
            forwards.insertString("class QDomElement");
            if (!classData.signals.isEmpty())
                forwards.insertString("class QString");
            break;

        case Class::Proxy:
            includes["Qt"].insertString("<qobject.h>");
            forwards.insertString("class QDBusConnection");
            forwards.insertString("class QDBusError");
            forwards.insertString("class QDBusMessage");
            forwards.insertString("class QDBusProxy");
            forwards.insertString("class QString");
            if (!classData.properties.isEmpty())
                forwards.insertString("class QDBusVariant");
            break;

        case Class::Node:
            includes["qdbus"].insertString("<dbus/qdbusobject.h>");
            forwards.insertString("class QDBusConnection");
            forwards.insertString("class QString");
            break;
    }

    if (!includes["Qt"].isEmpty())
        writeIncludes("Qt", includes["Qt"].keys(), stream);

    if (!includes["qdbus"].isEmpty())
        writeIncludes("Qt D-Bus", includes["qdbus"].keys(), stream);

    if (!includes["local"].isEmpty())
        writeIncludes("local", includes["local"].keys(), stream);

    stream << "// forward declarations" << endl;
    Set::const_iterator setIt    = forwards.begin();
    Set::const_iterator setEndIt = forwards.end();
    for (; setIt != setEndIt; ++setIt)
    {
        stream << setIt.key() << ";" << endl;
    }
    stream << endl;
}

static void extractSourceIncludes(const Method& method,
        QMap<QString, Set>& includes)
{
    QValueList<Argument>::const_iterator it    = method.arguments.begin();
    QValueList<Argument>::const_iterator endIt = method.arguments.end();
    for (; it != endIt; ++it)
    {
        if ((*it).sourceIncludes.isEmpty()) continue;

        QMap<QString, QStringList>::const_iterator mapIt =
            (*it).sourceIncludes.begin();
        QMap<QString, QStringList>::const_iterator mapEndIt =
            (*it).sourceIncludes.end();

        for (; mapIt != mapEndIt; ++mapIt)
        {
            includes[mapIt.key()].insertStringList(mapIt.data());
        }
    }
}

static void writeSourceIncludes(const Class& classData, Class::Role role,
        QTextStream& stream)
{
    QMap<QString, Set> includes;

    QValueList<Method>::const_iterator it    = classData.methods.begin();
    QValueList<Method>::const_iterator endIt = classData.methods.end();
    for (; it != endIt; ++it)
    {
        if ((*it).arguments.isEmpty()) continue;

        extractSourceIncludes(*it, includes);
    }

    it    = classData.signals.begin();
    endIt = classData.signals.end();
    for (; it != endIt; ++it)
    {
        if ((*it).arguments.isEmpty()) continue;

        extractSourceIncludes(*it, includes);
    }

    QValueList<Property>::const_iterator propertyIt = classData.properties.begin();
    QValueList<Property>::const_iterator propertyEndIt = classData.properties.end();
    for (; propertyIt != propertyEndIt; ++propertyIt)
    {
        if ((*propertyIt).sourceIncludes.isEmpty()) continue;

        QMap<QString, QStringList>::const_iterator mapIt =
            (*propertyIt).sourceIncludes.begin();
        QMap<QString, QStringList>::const_iterator mapEndIt =
            (*propertyIt).sourceIncludes.end();

        for (; mapIt != mapEndIt; ++mapIt)
        {
            includes[mapIt.key()].insertStringList(mapIt.data());
        }
    }

    switch (role)
    {
        case Class::Interface:
            includes["Qt"].insertString("<qdom.h>");
            includes["qdbus"].insertString("<dbus/qdbuserror.h>");
            includes["qdbus"].insertString("<dbus/qdbusmessage.h>");
            break;

        case Class::Proxy:
            includes["qdbus"].insertString("<dbus/qdbuserror.h>");
            includes["qdbus"].insertString("<dbus/qdbusmessage.h>");
            includes["qdbus"].insertString("<dbus/qdbusproxy.h>");
            if (!classData.properties.isEmpty())
            {
                includes["qdbus"].insertString("<dbus/qdbusconnection.h>");
                includes["qdbus"].insertString("<dbus/qdbusvariant.h>");
            }
            break;

        case Class::Node:
            includes["Qt"].insertString("<qdom.h>");
            includes["Qt"].insertString("<qmap.h>");
            includes["qdbus"].insertString("<dbus/qdbusconnection.h>");
            includes["qdbus"].insertString("<dbus/qdbusmessage.h>");
            break;
    }

    if (!includes["Qt"].isEmpty())
        writeIncludes("Qt", includes["Qt"].keys(), stream);

    if (!includes["qdbus"].isEmpty())
        writeIncludes("Qt D-Bus", includes["qdbus"].keys(), stream);

    if (!includes["local"].isEmpty())
        writeIncludes("local", includes["local"].keys(), stream);

    stream << endl;
}

static void writeInterfaceIncludes(const QValueList<Class> interfaces,
        QTextStream& stream)
{
    stream << "// interface classes includes" << endl;

    QValueList<Class>::const_iterator it    = interfaces.begin();
    QValueList<Class>::const_iterator endIt = interfaces.end();
    for (; it != endIt; ++it)
    {
        stream << "#include \"" << (*it).name.lower() << ".h\"" << endl;
    }

    stream << "#include \"introspectable.h\"" << endl;

    stream << endl;
}

static void openClassDeclaration(const Class& classData,
    Class::Role role, QTextStream& stream)
{
    switch (role)
    {
        case Class::Interface:
            stream << "class " << classData.name << " : public QDBusObjectBase"
                   << endl;
            stream << "{" << endl;
            stream << "public:" << endl;
            stream << "    virtual ~" << classData.name << "() {}" << endl;
            stream << endl;
            stream << "    static void buildIntrospectionData(QDomElement& interfaceElement);" << endl;
            break;

        case Class::Proxy:
            stream << "class " << classData.name << " : public QObject" << endl;
            stream << "{" << endl;
            stream << "    Q_OBJECT" << endl;
            stream << "public:" << endl;
            stream << "    " << classData.name
                   << "(const QString& service, const QString& path, QObject* parent = 0, const char* name = 0);" << endl;
            stream << endl;

            stream << "    virtual ~" << classData.name << "();" << endl;
            stream << endl;

            stream << "    void setConnection(const QDBusConnection& connection);"
                   << endl;
            break;

        case Class::Node:
            stream << "class " << classData.name << " : public QDBusObjectBase"
                   << endl;
            stream << "{" << endl;
            stream << "public:" << endl;
            stream << "    " << classData.name << "();" << endl;
            stream << endl;
            stream << "    virtual ~" << classData.name << "();" << endl;
            stream << endl;
            stream << "    bool registerObject(const QDBusConnection& connection, "
                   << "const QString& path);" << endl;
            stream << endl;
            stream << "    void unregisterObject();" << endl;
            stream << endl;
            stream << "protected:" << endl;
            stream << "    virtual QDBusObjectBase* createInterface("
                   << "const QString& interfaceName) = 0;" << endl;
            stream << endl;
            stream << "protected: // usually no need to reimplement" << endl;
            stream << "    virtual bool handleMethodCall(const QDBusMessage& message);" << endl;
            stream << endl;
            stream << "private:" << endl;
            stream << "    class Private;" << endl;
            stream << "    Private* m_private;" << endl;
            break;
    }

    stream << endl;
}

static void closeClassDeclaration(const Class& classData, Class::Role role,
        QTextStream& stream)
{
    switch (role)
    {
        case Class::Interface:
            break;

        case Class::Proxy:
            stream << "private: // Hiding copy constructor and assignment operator" << endl;
            stream << "    " << classData.name << "(const "
                   << classData.name << "&);" << endl;
            stream << "    " << classData.name << "& operator=(const "
                   << classData.name << "&);" << endl;
            break;

        case Class::Node:
            stream << "private: // Hiding copy constructor and assignment operator" << endl;
            stream << "    " << classData.name << "(const "
                   << classData.name << "&);" << endl;
            stream << "    " << classData.name << "& operator=(const "
                   << classData.name << "&);" << endl;
            break;
    }
    stream << "}; // class " << classData.name << endl;
    stream << endl;
}

static void writeMethodDeclarations(const Class& classData, Class::Role role,
        QTextStream& stream)
{
    if (!classData.methods.isEmpty())
    {
        bool pureVirtual = true;
        switch (role)
        {
            case Class::Interface:
                pureVirtual = true;
                stream << "protected:" << endl;
                break;

            case Class::Proxy:
                pureVirtual = false;
                stream << "public:" << endl;
                break;

            case Class::Node: // no variable methods
                break;
        }

        QValueList<Method>::const_iterator it    = classData.methods.begin();
        QValueList<Method>::const_iterator endIt = classData.methods.end();
        for (; it != endIt; ++it)
        {
            stream << "    virtual bool ";
            MethodGenerator::writeMethodDeclaration(*it, pureVirtual, true, stream);
        }
    }

    if (!classData.properties.isEmpty())
    {
        bool pureVirtual = true;
        bool skip = false;
        switch (role)
        {
            case Class::Interface:
                qWarning("Properties not yet supported for interfaces");
                skip = true;
                pureVirtual = true;
                break;

            case Class::Proxy:
                pureVirtual = false;
                stream << "public:" << endl;
                stream << "    virtual void setDBusProperty(const QString& name,"
                       << " const QDBusVariant& variant, QDBusError& error);"
                       << endl;
                stream << "    virtual QDBusVariant getDBusProperty(const QString& name, QDBusError& error) const;" << endl;
                stream << endl;
                break;

            case Class::Node: // no node properties
                skip = true;
                break;
        }

        if (!skip)
        {
            QValueList<Property>::const_iterator it    = classData.properties.begin();
            QValueList<Property>::const_iterator endIt = classData.properties.end();
            for (; it != endIt; ++it)
            {
                MethodGenerator::writePropertyDeclaration(*it, pureVirtual, stream);
            }
        }
    }

    switch (role)
    {
        case Class::Interface:
            if (!classData.methods.isEmpty())
            {
                stream << "protected: // implement sending replies" << endl;
                stream << "    virtual void handleMethodReply(const QDBusMessage& reply) = 0;" << endl;
                stream << endl;
                stream << "protected: // usually no need to reimplement" << endl;
                stream << "    virtual bool handleMethodCall(const QDBusMessage& message);" << endl;
            }
            else
            {
                stream << "protected: // no methods to handle" << endl;
                stream << "    virtual bool handleMethodCall(const QDBusMessage&) { return false; }" << endl;
            }
            break;

        case Class::Proxy:
            if (!classData.signals.isEmpty())
            {
                stream << "protected slots: // usually no need to reimplement" << endl;
                stream << "    virtual void slotHandleDBusSignal(const QDBusMessage& message);" << endl;
                stream << endl;
            }
            stream << "protected:" << endl;
            stream << "    QDBusProxy* m_baseProxy;" << endl;
            break;

        case Class::Node: // not variable methods
            break;
    }

    stream << endl;
}

static void writeSignalDeclarations(const Class& classData, Class::Role role,
        QTextStream& stream)
{
    if (classData.signals.isEmpty()) return;

    QString prefix;
    switch (role)
    {
        case Class::Interface:
            stream << "protected: // implement sending signals" << endl;
            stream << "    virtual bool handleSignalSend(const QDBusMessage& reply) = 0;" << endl;
            stream << "    virtual QString objectPath() const = 0;" << endl;
            stream << endl;
            stream << "protected: // for sending D-Bus signals" << endl;
            prefix = "    virtual bool emit";
            break;

        case Class::Proxy:
            stream << "signals:" << endl;
            prefix = "    void ";
            break;

        case Class::Node: // no signals
            break;
    }

    QValueList<Method>::const_iterator it    = classData.signals.begin();
    QValueList<Method>::const_iterator endIt = classData.signals.end();
    for (; it != endIt; ++it)
    {
        stream << prefix;
        MethodGenerator::writeMethodDeclaration(*it, false, false, stream);
    }

    stream << endl;
}

static void writeSignalEmitters(const Class& classData, QTextStream& stream)
{
    if (classData.signals.isEmpty()) return;

    QValueList<Method>::const_iterator it    = classData.signals.begin();
    QValueList<Method>::const_iterator endIt = classData.signals.end();
    for (; it != endIt; ++it)
    {
        MethodGenerator::writeSignalEmitter(classData, *it, stream);
    }

    stream << endl;
}

static void writeMethodCallDeclarations(const Class& classData,
        QTextStream& stream)
{
    if (classData.methods.isEmpty()) return;

    QValueList<Method>::const_iterator it    = classData.methods.begin();
    QValueList<Method>::const_iterator endIt = classData.methods.end();
    for (; it != endIt; ++it)
    {
        stream << "    ";
        MethodGenerator::writeMethodCallDeclaration(*it, stream);
    }
}

static void writeMethodCalls(const Class& classData, QTextStream& stream)
{
    QValueList<Method>::const_iterator it    = classData.methods.begin();
    QValueList<Method>::const_iterator endIt = classData.methods.end();
    for (; it != endIt; ++it)
    {
        MethodGenerator::writeMethodCall(classData, *it, stream);
    }
}

static void writeProxyMethods(const Class& classData, QTextStream& stream)
{
    QValueList<Method>::const_iterator it    = classData.methods.begin();
    QValueList<Method>::const_iterator endIt = classData.methods.end();
    for (; it != endIt; ++it)
    {
        MethodGenerator::writeProxyMethod(classData.name, *it, stream);
    }
}

static void writeProxyProperties(const Class& classData, QTextStream& stream)
{
    if (classData.properties.isEmpty()) return;

    MethodGenerator::writeProxyGenericProperty(classData, stream);

    QValueList<Property>::const_iterator it    = classData.properties.begin();
    QValueList<Property>::const_iterator endIt = classData.properties.end();
    for (; it != endIt; ++it)
    {
        MethodGenerator::writeProxyProperty(classData, *it, stream);
    }
}

bool ClassGenerator::initStreams(const QString& baseName,
                                 QTextStream& headerStream,
                                 QTextStream& sourceStream)
{
    QFile* headerFile = new QFile(baseName + ".h");
    QFile* sourceFile = new QFile(baseName + ".cpp");

    if (!headerFile->open(IO_WriteOnly) || !sourceFile->open(IO_WriteOnly))
    {
        delete headerFile;
        delete sourceFile;

        return false;
    }

    headerStream.setDevice(headerFile);
    sourceStream.setDevice(sourceFile);

    // create header
    writeFileHeader(headerStream);
    openIncludeGuard(baseName, headerStream);

    // create source
    writeFileHeader(sourceStream);
    sourceStream << "// declaration include" << endl;
    sourceStream << "#include \"" << baseName << ".h\"" << endl;
    sourceStream << endl;

    return true;
}

bool ClassGenerator::finishStreams(const QString& baseName,
                                   QTextStream& headerStream,
                                   QTextStream& sourceStream)
{
    closeIncludeGuard(baseName, headerStream);
    writeFileFooter(headerStream);

    writeFileFooter(sourceStream);

    QIODevice* device = headerStream.device();
    headerStream.unsetDevice();
    delete device;

    device = sourceStream.device();
    sourceStream.unsetDevice();
    delete device;

    return true;
}

bool ClassGenerator::extractClass(const QDomElement& interfaceElement,
                                   Class& classData)
{
    qDebug("ClassGenerator: processing interface '%s'",
           interfaceElement.attribute("name").latin1());

    classData.dbusName = interfaceElement.attribute("name");

    QStringList nameParts = QStringList::split('.', classData.dbusName);

    if (nameParts.count() < 2) return false;

    classData.name = nameParts.back();
    nameParts.pop_back();
    classData.namespaces = nameParts;

    return MethodGenerator::extractMethods(interfaceElement, classData);
}

bool ClassGenerator::generateInterface(const Class& classData,
                                       QTextStream& headerStream,
                                       QTextStream& sourceStream)
{
    // create header
    writeHeaderIncludes(classData, Class::Interface, headerStream);

    openNamespaces(classData.namespaces, headerStream);
    openClassDeclaration(classData, Class::Interface, headerStream);

    writeSignalDeclarations(classData, Class::Interface, headerStream);
    writeMethodDeclarations(classData, Class::Interface, headerStream);
    writeMethodCallDeclarations(classData, headerStream);

    closeClassDeclaration(classData, Class::Interface, headerStream);
    closeNamespaces(classData.namespaces, headerStream);

    // create source
    writeSourceIncludes(classData, Class::Interface, sourceStream);

    openNamespaces(classData.namespaces, sourceStream);

    MethodGenerator::writeIntrospectionDataMethod(classData, sourceStream);

    writeSignalEmitters(classData, sourceStream);

    writeMethodCalls(classData, sourceStream);

    MethodGenerator::writeInterfaceMainMethod(classData, sourceStream);

    closeNamespaces(classData.namespaces, sourceStream);

    return true;
}

bool ClassGenerator::generateProxy(const Class& classData,
                                   QTextStream& headerStream,
                                   QTextStream& sourceStream)
{
    // create header
    writeHeaderIncludes(classData, Class::Proxy, headerStream);

    openNamespaces(classData.namespaces, headerStream);
    openClassDeclaration(classData, Class::Proxy, headerStream);

    writeSignalDeclarations(classData, Class::Proxy, headerStream);
    writeMethodDeclarations(classData, Class::Proxy, headerStream);

    closeClassDeclaration(classData, Class::Proxy, headerStream);
    closeNamespaces(classData.namespaces, headerStream);

    // create source
    writeSourceIncludes(classData, Class::Proxy, sourceStream);

    openNamespaces(classData.namespaces, sourceStream);

    MethodGenerator::writeProxyBegin(classData, sourceStream);

    writeProxyMethods(classData, sourceStream);

    writeProxyProperties(classData, sourceStream);

    if (!classData.signals.isEmpty())
        MethodGenerator::writeSignalHandler(classData, sourceStream);

    closeNamespaces(classData.namespaces, sourceStream);

    return true;
}

bool ClassGenerator::generateNode(const Class& classData,
                                  const QValueList<Class>& interfaces,
                                  QTextStream& headerStream,
                                  QTextStream& sourceStream)
{
    // create header
    writeHeaderIncludes(classData, Class::Node, headerStream);

    openNamespaces(classData.namespaces, headerStream);
    openClassDeclaration(classData, Class::Node, headerStream);

    closeClassDeclaration(classData, Class::Node, headerStream);
    closeNamespaces(classData.namespaces, headerStream);
    closeIncludeGuard(classData.name, headerStream);

    // create source
    writeSourceIncludes(classData, Class::Node, sourceStream);
    writeInterfaceIncludes(interfaces, sourceStream);

    openNamespaces(classData.namespaces, sourceStream);

    MethodGenerator::writeNodePrivate(classData, sourceStream);

    MethodGenerator::writeNodeBegin(classData, sourceStream);

    MethodGenerator::writeNodeMethods(classData, interfaces, sourceStream);

    closeNamespaces(classData.namespaces, sourceStream);

    return true;
}

// End of File
